unit Sphinx.EmbeddedBrowser;

interface

uses
  Winapi.Windows, Winapi.Messages, System.SysUtils, System.Variants, System.Classes, Vcl.Graphics,
  Vcl.Controls, Vcl.Forms, Vcl.Dialogs, Winapi.WebView2, Winapi.ActiveX, Vcl.Edge,
  Sparkle.Uri, Sphinx.Browser;

type
  TFormEmbeddedBrowser = class(TInterfacedObject, IBrowser)
  strict private
    FOptions: TBrowserOptions;
    FForm: TForm;
    FBrowser: TEdgeBrowser;
    FResponseUrl: string;
    function GetForm: TForm;
    function GetBrowser: TEdgeBrowser;
    procedure FormDestroy(Sender: TObject);
    procedure BrowserNavigationStarting(Sender: TCustomEdgeBrowser; Args: TNavigationStartingEventArgs);
    procedure FormClose(Sender: TObject; var Action: TCloseAction);
  strict protected
    property Form: TForm read GetForm;
    property Browser: TEdgeBrowser read GetBrowser;
  public
    constructor Create;
    destructor Destroy; override;
    procedure Execute(Callback: TBrowserCallback);
    function DefaultRedirectUrl: string;
    function Options: TBrowserOptions;
  end;

implementation

{ TFormEmbeddedBrowser }

procedure TFormEmbeddedBrowser.BrowserNavigationStarting(Sender: TCustomEdgeBrowser; Args: TNavigationStartingEventArgs);
var
  Uri: IUri;
  TargetUrl: PChar;
  EndUri: IUri;
begin
  Args.ArgsInterface.Get_uri(TargetUrl);
  Uri := TUri.Create(TargetUrl);
  EndUri := TUri.Create(Options.EndUrl);
  if SameText(Uri.Host, EndUri.Host) and SameText(Uri.AbsolutePath, EndUri.AbsolutePath) then
  begin
    FResponseUrl := TargetUrl;
    Args.ArgsInterface.Set_Cancel(1);
    Form.Close;
  end;
end;

constructor TFormEmbeddedBrowser.Create;
begin
  inherited Create;
  FOptions := TBrowserOptions.Create;
end;

function TFormEmbeddedBrowser.DefaultRedirectUrl: string;
begin
  Result := 'http://127.0.0.1/signin-embedded';
end;

destructor TFormEmbeddedBrowser.Destroy;
begin
  FForm.Free;
  FOptions.Free;
  inherited;
end;

procedure TFormEmbeddedBrowser.Execute(Callback: TBrowserCallback);
var
  Response: TBrowserResponse;
begin
  FResponseUrl := '';
  Browser.Navigate(Options.StartUrl);
  Form.ShowModal;
  if FResponseUrl <> '' then
  begin
    Response := TBrowserResponse.Create;
    try
      Response.Response := FResponseUrl;
      if Assigned(Callback) then
        Callback(Response);
    finally
      Response.Free;
    end;
  end;
end;

procedure TFormEmbeddedBrowser.FormClose(Sender: TObject; var Action: TCloseAction);
begin
  Action := caFree;
end;

procedure TFormEmbeddedBrowser.FormDestroy(Sender: TObject);
begin
  FForm := nil;
end;

function TFormEmbeddedBrowser.GetBrowser: TEdgeBrowser;
begin
  GetForm;
  Result := FBrowser;
end;

function TFormEmbeddedBrowser.GetForm: TForm;
begin
  if FForm = nil then
  begin
    FForm := TForm.Create(Application);
    FForm.OnDestroy := FormDestroy;
    FForm.OnClose := FormClose;
    FForm.Caption := 'Login';
    FForm.Position := poMainFormCenter;
    FBrowser := TEdgeBrowser.Create(FForm);
    FBrowser.Parent := FForm;
    FBrowser.Align := alClient;
    FBrowser.OnNavigationStarting := BrowserNavigationStarting;
    FForm.ClientWidth := MulDiv(800, Screen.PixelsPerInch, 96);
    FForm.ClientHeight := MulDiv(600, Screen.PixelsPerInch, 96);
  end;
  Result := FForm;
end;

function TFormEmbeddedBrowser.Options: TBrowserOptions;
begin
  Result := FOptions;
end;

end.


